#include "DXUT.h"
#include "SDKmisc.h"
#include "DXUTgui.h" 
#include <d3dcompiler.h>
#pragma warning( disable : 4100 )

#include "dpLib.h"
#include "dpUtils.h"
#include <string>
#include <filesystem>
#include <fstream>
#include <vector>

//--------------------------------------------------------------------------------------
// Structures
//--------------------------------------------------------------------------------------
struct SimpleVertex
{
	DirectX::XMFLOAT3 Pos;
	DirectX::XMFLOAT2 Tex;
};

struct CBChangesEveryFrame
{
	DirectX::XMFLOAT4X4 mWorldViewProj;
};

//--------------------------------------------------------------------------------------
// Global Variables
//--------------------------------------------------------------------------------------
ID3D11VertexShader*         g_pVertexShader = nullptr;
ID3D11PixelShader*          g_pPixelShader = nullptr;
ID3D11InputLayout*          g_pVertexLayout = nullptr;
ID3D11Buffer*               g_pVertexBuffer = nullptr;
ID3D11Buffer*               g_pIndexBuffer = nullptr;
ID3D11Buffer*               g_pCBChangesEveryFrame = nullptr;
ID3D11ShaderResourceView*   g_pSkyboxeRV[6];
ID3D11SamplerState*         g_pSamplerLinear = nullptr;
ID3D11Texture2D *           g_pResource = nullptr;
DirectX::XMMATRIX           g_World;
DirectX::XMMATRIX           g_View;
DirectX::XMMATRIX           g_Projection;
D3D11_VIEWPORT              g_Viewport;

CDXUTDialogResourceManager  g_DialogResourceManager; // manager for shared resources of dialogs 
CDXUTTextHelper*            g_pTxtHelper = nullptr;

int                         g_windowPosX = 0;
int                         g_windowPosY = 0;
int                         g_windowWidth = 1000;
int                         g_windowHeight = 1000;
std::string g_configfile = "data\\config.xml";
int                         g_channel = 0;

dpContext*                  CTX = NULL;
dpVec3f                     POSITION(0, 0, 0);

float                       g_fHeading = 0.f;
float                       g_fPitch = 0.f;
float                       g_fRoll = 0.f;

float                       g_fWarping = 1.f;
float                       g_fBlending = 1.f;
float                       g_fBlackLevel = 1.f;
bool                        g_bWarping = true;
bool                        g_bBlending = true;
bool                        g_bBlackLevel = true;
bool                        g_bSecondaryBlending = true;

HRESULT WINAPI D3DCompileFromFile( std::filesystem::path file,
						 _In_reads_opt_( _Inexpressible_( pDefines->Name != NULL ) ) CONST D3D_SHADER_MACRO* pDefines,
						 _In_opt_ ID3DInclude* pInclude,
						 _In_ LPCSTR pEntrypoint,
						 _In_ LPCSTR pTarget,
						 _In_ UINT Flags1,
						 _In_ UINT Flags2,
						 _Out_ ID3DBlob** ppCode,
						 _Always_( _Outptr_opt_result_maybenull_ ) ID3DBlob** ppErrorMsgs ) {
	if( std::ifstream ifs( file, std::ios::ate | std::ios::binary ); ifs.is_open() ) {
		std::vector<char> data( ifs.tellg(), '\0' );
		ifs.seekg( 0 );
		ifs.read( data.data(), data.size() );
		return D3DCompile( data.data(), data.size(), file.filename().string().c_str(), pDefines, pInclude, pEntrypoint, pTarget, Flags1, Flags2, ppCode, ppErrorMsgs );
	}
	if( ppErrorMsgs )
		*ppErrorMsgs = nullptr;
	return E_FAIL;
}
//--------------------------------------------------------------------------------------
// Reject any D3D11 devices that aren't acceptable by returning false
//--------------------------------------------------------------------------------------
bool CALLBACK IsD3D11DeviceAcceptable(const CD3D11EnumAdapterInfo *AdapterInfo, UINT Output, const CD3D11EnumDeviceInfo *DeviceInfo, DXGI_FORMAT BackBufferFormat, bool bWindowed, void* pUserContext)
{
    if(BackBufferFormat == DXGI_FORMAT_R8G8B8A8_UNORM)
        return true;
    return false;
}

//--------------------------------------------------------------------------------------
// Called right before creating a D3D device, allowing the app to modify the device settings as needed
//--------------------------------------------------------------------------------------
bool CALLBACK ModifyDeviceSettings(DXUTDeviceSettings* pDeviceSettings, void* pUserContext)
{
	//pDeviceSettings->d3d11.sd.BufferUsage |= DXGI_USAGE_SHADER_INPUT;
	return true;
}

// --------------------------------------------------------------------------------------
// Create any D3D11 resources that aren't dependant on the back buffer
//--------------------------------------------------------------------------------------
HRESULT CALLBACK OnD3D11CreateDevice(ID3D11Device* pd3dDevice, const DXGI_SURFACE_DESC* pBackBufferSurfaceDesc, void* pUserContext)
{
	HRESULT hr = S_OK;
	auto pd3dImmediateContext = DXUTGetD3D11DeviceContext();

	V_RETURN(g_DialogResourceManager.OnD3D11CreateDevice(pd3dDevice, pd3dImmediateContext));
	g_pTxtHelper = new CDXUTTextHelper(pd3dDevice, pd3dImmediateContext, &g_DialogResourceManager, 100);


	DWORD dwShaderFlags = D3DCOMPILE_ENABLE_STRICTNESS;
#ifdef _DEBUG
	// Set the D3DCOMPILE_DEBUG flag to embed debug information in the shaders.
	// Setting this flag improves the shader debugging experience, but still allows 
	// the shaders to be optimized and to run exactly the way they will run in 
	// the release configuration of this program.
	dwShaderFlags |= D3DCOMPILE_DEBUG;

	// Disable optimizations to further improve shader debugging
	dwShaderFlags |= D3DCOMPILE_SKIP_OPTIMIZATION;
#endif

	// Compile the vertex shader
	ID3DBlob* pVSBlob = nullptr;
	V_RETURN(D3DCompileFromFile(L"shader.fx", nullptr, nullptr, "VS", "vs_4_0", dwShaderFlags, 0, &pVSBlob, nullptr));

	// Create the vertex shader
	hr = pd3dDevice->CreateVertexShader(pVSBlob->GetBufferPointer(), pVSBlob->GetBufferSize(), nullptr, &g_pVertexShader);
	if (FAILED(hr))
	{
		SAFE_RELEASE(pVSBlob);
		return hr;
	}

	// Define the input layout
	D3D11_INPUT_ELEMENT_DESC layout[] =
	{
		{ "POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0 },
		{ "TEXCOORD", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 12, D3D11_INPUT_PER_VERTEX_DATA, 0 },
	};
	UINT numElements = ARRAYSIZE(layout);

	// Create the input layout
	hr = pd3dDevice->CreateInputLayout(layout, numElements, pVSBlob->GetBufferPointer(), pVSBlob->GetBufferSize(), &g_pVertexLayout);
	SAFE_RELEASE(pVSBlob);
	if (FAILED(hr))
		return hr;

	// Set the input layout
	pd3dImmediateContext->IASetInputLayout(g_pVertexLayout);

	// Compile the pixel shader
	ID3DBlob* pPSBlob = nullptr;
	V_RETURN(D3DCompileFromFile(L"shader.fx", nullptr, nullptr, "PS", "ps_4_0", dwShaderFlags, 0, &pVSBlob, nullptr));

	// Create the pixel shader
	hr = pd3dDevice->CreatePixelShader(pPSBlob->GetBufferPointer(), pPSBlob->GetBufferSize(), nullptr, &g_pPixelShader);
	SAFE_RELEASE(pPSBlob);
	if (FAILED(hr))
		return hr;

	float w = 10.0;
	float h = 10.0;
	float d = 10.0;

	// Create vertex buffer
	SimpleVertex vertices[] =
	{
		// front
		{ DirectX::XMFLOAT3(-w / 2.0f, -h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(0.0f, 1.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f, -h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(1.0f, 1.0f) },
		{ DirectX::XMFLOAT3(-w / 2.0f,  h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(0.0f, 0.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f,  h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(1.0f, 0.0f) },

		// back
		{ DirectX::XMFLOAT3(w / 2.0f, -h / 2.0f, d / 2.0f), DirectX::XMFLOAT2(0.0f, 1.0f) },
		{ DirectX::XMFLOAT3(-w / 2.0f, -h / 2.0f, d / 2.0f), DirectX::XMFLOAT2(1.0f, 1.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f,  h / 2.0f, d / 2.0f), DirectX::XMFLOAT2(0.0f, 0.0f) },
		{ DirectX::XMFLOAT3(-w / 2.0f,  h / 2.0f, d / 2.0f), DirectX::XMFLOAT2(1.0f, 0.0f) },

		// left
		{ DirectX::XMFLOAT3(-w / 2.0f, -h / 2.0f,  d / 2.0f), DirectX::XMFLOAT2(0.0f, 1.0f) },
		{ DirectX::XMFLOAT3(-w / 2.0f, -h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(1.0f, 1.0f) },
		{ DirectX::XMFLOAT3(-w / 2.0f,  h / 2.0f,  d / 2.0f), DirectX::XMFLOAT2(0.0f, 0.0f) },
		{ DirectX::XMFLOAT3(-w / 2.0f,  h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(1.0f, 0.0f) },

		// right
		{ DirectX::XMFLOAT3(w / 2.0f, -h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(0.0f, 1.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f, -h / 2.0f,  d / 2.0f), DirectX::XMFLOAT2(1.0f, 1.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f,  h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(0.0f, 0.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f,  h / 2.0f,  d / 2.0f), DirectX::XMFLOAT2(1.0f, 0.0f) },

		// bottom
		{ DirectX::XMFLOAT3(-w / 2.0f, -h / 2.0f,  d / 2.0f), DirectX::XMFLOAT2(0.0f, 1.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f, -h / 2.0f,  d / 2.0f), DirectX::XMFLOAT2(1.0f, 1.0f) },
		{ DirectX::XMFLOAT3(-w / 2.0f, -h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(0.0f, 0.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f, -h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(1.0f, 0.0f) },

		// top
		{ DirectX::XMFLOAT3(-w / 2.0f, h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(0.0f, 1.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f, h / 2.0f, -d / 2.0f), DirectX::XMFLOAT2(1.0f, 1.0f) },
		{ DirectX::XMFLOAT3(-w / 2.0f, h / 2.0f,  d / 2.0f), DirectX::XMFLOAT2(0.0f, 0.0f) },
		{ DirectX::XMFLOAT3(w / 2.0f, h / 2.0f,  d / 2.0f), DirectX::XMFLOAT2(1.0f, 0.0f) },
	};

	D3D11_BUFFER_DESC bd;
	ZeroMemory(&bd, sizeof(bd));
	bd.Usage = D3D11_USAGE_DEFAULT;
	bd.ByteWidth = sizeof(SimpleVertex) * 24;
	bd.BindFlags = D3D11_BIND_VERTEX_BUFFER;
	bd.CPUAccessFlags = 0;
	D3D11_SUBRESOURCE_DATA InitData;
	ZeroMemory(&InitData, sizeof(InitData));
	InitData.pSysMem = vertices;
	V_RETURN(pd3dDevice->CreateBuffer(&bd, &InitData, &g_pVertexBuffer));

	// Set vertex buffer
	UINT stride = sizeof(SimpleVertex);
	UINT offset = 0;
	pd3dImmediateContext->IASetVertexBuffers(0, 1, &g_pVertexBuffer, &stride, &offset);

	// Create index buffer
	DWORD indices[] =
	{
		0,1,2,
		1,3,2,

		4,5,6,
		5,7,6,

		8,9,10,
		9,11,10,

		12,13,14,
		13,15,14,

		16,17,18,
		17,19,18,

		20,21,22,
		21,23,22
	};

	bd.Usage = D3D11_USAGE_DEFAULT;
	bd.ByteWidth = sizeof(DWORD) * 36;
	bd.BindFlags = D3D11_BIND_INDEX_BUFFER;
	bd.CPUAccessFlags = 0;
	bd.MiscFlags = 0;
	InitData.pSysMem = indices;
	V_RETURN(pd3dDevice->CreateBuffer(&bd, &InitData, &g_pIndexBuffer));

	// Set index buffer
	pd3dImmediateContext->IASetIndexBuffer(g_pIndexBuffer, DXGI_FORMAT_R32_UINT, 0);

	// Set primitive topology
	pd3dImmediateContext->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST);

	// Create the constant buffers
	bd.Usage = D3D11_USAGE_DYNAMIC;
	bd.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	bd.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	bd.ByteWidth = sizeof(CBChangesEveryFrame);
	V_RETURN(pd3dDevice->CreateBuffer(&bd, nullptr, &g_pCBChangesEveryFrame));

	// Initialize the world matrices
	g_World = DirectX::XMMatrixIdentity();

	// Initialize the view matrix
	static const DirectX::XMVECTORF32 s_Eye = { 0.0f, 0.0f, 0.0f, 0.f };
	static const DirectX::XMVECTORF32 s_At = { 0.0f, 0.0f, -1.0f, 0.f };
	static const DirectX::XMVECTORF32 s_Up = { 0.0f, 1.0f, 0.0f, 0.f };
	g_View = DirectX::XMMatrixLookAtLH(s_Eye, s_At, s_Up);

	g_World = DirectX::XMMatrixIdentity();

	// Create the sample state
	D3D11_SAMPLER_DESC sampDesc;
	ZeroMemory(&sampDesc, sizeof(sampDesc));
	sampDesc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
	sampDesc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
	sampDesc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
	sampDesc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
	sampDesc.ComparisonFunc = D3D11_COMPARISON_NEVER;
	sampDesc.MinLOD = 0;
	sampDesc.MaxLOD = D3D11_FLOAT32_MAX;
	V_RETURN(pd3dDevice->CreateSamplerState(&sampDesc, &g_pSamplerLinear));

	// Load the skybox textures
	V_RETURN(DXUTCreateShaderResourceViewFromFile(pd3dDevice, L"data/cubemap/front.png", &g_pSkyboxeRV[0]));
	V_RETURN(DXUTCreateShaderResourceViewFromFile(pd3dDevice, L"data/cubemap/back.png", &g_pSkyboxeRV[1]));
	V_RETURN(DXUTCreateShaderResourceViewFromFile(pd3dDevice, L"data/cubemap/left.png", &g_pSkyboxeRV[2]));
	V_RETURN(DXUTCreateShaderResourceViewFromFile(pd3dDevice, L"data/cubemap/right.png", &g_pSkyboxeRV[3]));
	V_RETURN(DXUTCreateShaderResourceViewFromFile(pd3dDevice, L"data/cubemap/bottom.png", &g_pSkyboxeRV[4]));
	V_RETURN(DXUTCreateShaderResourceViewFromFile(pd3dDevice, L"data/cubemap/top.png", &g_pSkyboxeRV[5]));

	auto dpErr = dpCreateContextD3D11(&CTX, pd3dDevice, 2);
	if( dpNoError != dpErr ) {
		return E_FAIL;
	}
	
    dpSetFlipWarpmeshVerticesYD3D11(CTX, false);
	dpSetFlipWarpmeshTexcoordsVD3D11(CTX, false);
	dpErr = dpLoadConfigurationFromFileD3D11(CTX, g_configfile.c_str());
	if( dpNoError != dpErr ) {
		return E_FAIL;
	}
	dpSetClippingPlanesD3D11(CTX, 0.1f, 1000.0f);

	D3D11_TEXTURE2D_DESC desc;
	ZeroMemory(&desc, sizeof(D3D11_TEXTURE2D_DESC));
	desc.ArraySize = 1;
	desc.Format = pBackBufferSurfaceDesc->Format;
	desc.Width = pBackBufferSurfaceDesc->Width;
	desc.Height = pBackBufferSurfaceDesc->Height;
	desc.SampleDesc = pBackBufferSurfaceDesc->SampleDesc;
	desc.Usage = D3D11_USAGE_DEFAULT;
	desc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
	desc.CPUAccessFlags = 0;
	desc.MiscFlags = 0;
	desc.MipLevels = 1;
	hr = pd3dDevice->CreateTexture2D(&desc, nullptr, &g_pResource);

	return S_OK;
}

//--------------------------------------------------------------------------------------
// Create any D3D11 resources that depend on the back buffer
//--------------------------------------------------------------------------------------
HRESULT CALLBACK OnD3D11ResizedSwapChain(ID3D11Device* pd3dDevice, IDXGISwapChain* pSwapChain, const DXGI_SURFACE_DESC* pBackBufferSurfaceDesc, void* pUserContext)
{
	HRESULT hr;
	V_RETURN(g_DialogResourceManager.OnD3D11ResizedSwapChain(pd3dDevice, pBackBufferSurfaceDesc));

	// Setup the projection parameters
	float fAspect = static_cast<float>(pBackBufferSurfaceDesc->Width) / static_cast<float>(pBackBufferSurfaceDesc->Height);
	g_Projection = DirectX::XMMatrixPerspectiveFovRH(DirectX::XMConvertToRadians(60.f), fAspect, 0.1f, 100.0f);

    g_Viewport.Width = (float)pBackBufferSurfaceDesc->Width;
    g_Viewport.Height = (float)pBackBufferSurfaceDesc->Height;
    g_Viewport.MinDepth = 0.0f;
    g_Viewport.MaxDepth = 1.0f;
    g_Viewport.TopLeftX = 0.0f;
    g_Viewport.TopLeftY = 0.0f;

	return S_OK;
}

//--------------------------------------------------------------------------------------
// Handle updates to the scene.
//--------------------------------------------------------------------------------------
void CALLBACK OnFrameMove(double fTime, float fElapsedTime, void* pUserContext)
{
	DirectX::XMMATRIX mat, mat2, matY, matP, matR;
	matY = DirectX::XMMatrixRotationY(DirectX::XMConvertToRadians(g_fHeading));
	matP = DirectX::XMMatrixRotationX(DirectX::XMConvertToRadians(g_fPitch));
	matR = DirectX::XMMatrixRotationZ(DirectX::XMConvertToRadians(g_fRoll));
	mat2 = DirectX::XMMatrixMultiply(matP, matR);
	mat = DirectX::XMMatrixMultiply(matY, mat2);
	g_View = mat;
}

//--------------------------------------------------------------------------------------
// Render the scene using the D3D11 device
//--------------------------------------------------------------------------------------
void CALLBACK OnD3D11FrameRender(ID3D11Device* pd3dDevice, ID3D11DeviceContext* pd3dImmediateContext, double fTime, float fElapsedTime, void* pUserContext)
{
	dpMatrix4x4 projection;
	dpVec3f orientation;
	dpCamera camera;

	dpSetActiveChannelD3D11(CTX, g_channel, pd3dDevice, g_windowWidth, g_windowHeight);
    dpSetCorrectionPassD3D11_1(CTX, dpWarpingPass, g_fWarping);
    dpSetCorrectionPassD3D11_1(CTX, dpBlendingPass, g_fBlending);
    dpSetCorrectionPassD3D11_1(CTX, dpBlackLevelPass, g_fBlackLevel);
    dpSetCorrectionPassD3D11(CTX, dpWarpingPass, g_bWarping);
    dpSetCorrectionPassD3D11(CTX, dpBlendingPass, g_bBlending);
    dpSetCorrectionPassD3D11(CTX, dpBlackLevelPass, g_bBlackLevel);
    dpSetCorrectionPassD3D11(CTX, dpSecondaryBlendingPass, g_bSecondaryBlending);
    dpSetCorrectionPassD3D11(CTX, dpSecondaryBlendingPass, true);
    dpPreDrawD3D11(CTX, POSITION, &orientation, &projection);
	dpPreDrawD3D11_1(CTX, POSITION, &camera);

	//
	// Clear the back buffer
	//
	auto pRTV = DXUTGetD3D11RenderTargetView();
	pd3dImmediateContext->ClearRenderTargetView(pRTV, DirectX::Colors::Black);

	//
	// Clear the depth stencil
	//
	auto pDSV = DXUTGetD3D11DepthStencilView();
	pd3dImmediateContext->ClearDepthStencilView(pDSV, D3D11_CLEAR_DEPTH, 1.0, 0);

    // setting the viewport
    pd3dImmediateContext->RSSetViewports(1, &g_Viewport);

	//DirectX::XMMATRIX matViewRot;
	DirectX::XMMATRIX mat, mat2, matY, matP, matR;
	matY = DirectX::XMMatrixRotationY(DirectX::XMConvertToRadians(orientation.x));
	matP = DirectX::XMMatrixRotationX(DirectX::XMConvertToRadians(-orientation.y));
	matR = DirectX::XMMatrixRotationZ(DirectX::XMConvertToRadians(orientation.z));
	mat2 = DirectX::XMMatrixMultiply(matP, matR);
	mat = DirectX::XMMatrixMultiply(matY, mat2);
	//mat = DirectX::XMMatrixInverse(nullptr, mat);

	//DirectX::XMMATRIX mWorldViewProjection = g_World * mat * g_Projection;

    DirectX::XMMATRIX mProjection(projection.matrix);
    DirectX::XMMATRIX mWorldViewProjection = g_World * mat * mProjection;

	// Update constant buffer that changes once per frame
	HRESULT hr;
	D3D11_MAPPED_SUBRESOURCE MappedResource;
	V(pd3dImmediateContext->Map(g_pCBChangesEveryFrame, 0, D3D11_MAP_WRITE_DISCARD, 0, &MappedResource));
	auto pCB = reinterpret_cast<CBChangesEveryFrame*>(MappedResource.pData);
	XMStoreFloat4x4(&pCB->mWorldViewProj, DirectX::XMMatrixTranspose(mWorldViewProjection));
	pd3dImmediateContext->Unmap(g_pCBChangesEveryFrame, 0);

	D3D11_RASTERIZER_DESC rd;
	rd.CullMode = D3D11_CULL_NONE;
	ID3D11RasterizerState* pCurrentState;
	ID3D11RasterizerState* pState = nullptr;
	pd3dImmediateContext->RSGetState(&pCurrentState);
	pCurrentState->GetDesc(&rd);
	rd.CullMode = D3D11_CULL_NONE;
	
	pd3dDevice->CreateRasterizerState(&rd, &pState);
	pd3dImmediateContext->RSSetState(pState);

    pd3dImmediateContext->IASetInputLayout(g_pVertexLayout);

	//
	// Render the skybox
	//
	UINT stride = sizeof(SimpleVertex);
	UINT offset = 0;
	pd3dImmediateContext->IASetVertexBuffers(0, 1, &g_pVertexBuffer, &stride, &offset);
	pd3dImmediateContext->IASetIndexBuffer(g_pIndexBuffer, DXGI_FORMAT_R32_UINT, 0);
	pd3dImmediateContext->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST);
	pd3dImmediateContext->VSSetShader(g_pVertexShader, nullptr, 0);
	pd3dImmediateContext->VSSetConstantBuffers(0, 1, &g_pCBChangesEveryFrame);
	pd3dImmediateContext->PSSetShader(g_pPixelShader, nullptr, 0);
	pd3dImmediateContext->PSSetConstantBuffers(0, 1, &g_pCBChangesEveryFrame);
	pd3dImmediateContext->PSSetSamplers(0, 1, &g_pSamplerLinear);

	// front
	pd3dImmediateContext->PSSetShaderResources(0, 1, &g_pSkyboxeRV[0]);
	pd3dImmediateContext->DrawIndexed(6, 0, 0);
	// back
	pd3dImmediateContext->PSSetShaderResources(0, 1, &g_pSkyboxeRV[1]);
	pd3dImmediateContext->DrawIndexed(6, 6, 0);
	// right
	pd3dImmediateContext->PSSetShaderResources(0, 1, &g_pSkyboxeRV[2]);
	pd3dImmediateContext->DrawIndexed(6, 12, 0);
	// left
	pd3dImmediateContext->PSSetShaderResources(0, 1, &g_pSkyboxeRV[3]);
	pd3dImmediateContext->DrawIndexed(6, 18, 0);
	// bottom
	pd3dImmediateContext->PSSetShaderResources(0, 1, &g_pSkyboxeRV[4]);
	pd3dImmediateContext->DrawIndexed(6, 24, 0);
	// top
	pd3dImmediateContext->PSSetShaderResources(0, 1, &g_pSkyboxeRV[5]);
	pd3dImmediateContext->DrawIndexed(6, 30, 0);

	pd3dImmediateContext->RSSetState(pCurrentState);
	SAFE_RELEASE(pState);
	SAFE_RELEASE(pCurrentState);



	ID3D11Resource *resource = nullptr;
	DXUTGetD3D11RenderTargetView()->GetResource(&resource);
	pd3dImmediateContext->CopyResource(g_pResource, resource);
	ID3D11ShaderResourceView *srv = nullptr;
	hr = pd3dDevice->CreateShaderResourceView(g_pResource, NULL, &srv);
	//DXUTSaveTextureToFile(pd3dImmediateContext,  resource, false, L"rtv.bmp");
	resource->Release();
	
	dpPostDrawD3D11(CTX, srv, pd3dImmediateContext);

	srv->Release();

	g_pTxtHelper->Begin();
	g_pTxtHelper->SetInsertionPos(100, 100);
    DirectX::XMVECTORF32 gray = { 0.5f, 0.5f, 0.5f, 1.0f };
    g_pTxtHelper->SetForegroundColor(DirectX::Colors::Red);
	WCHAR message[200];
	swprintf(message, 200, L"EYEPOINT: (%.2f,%.2f,%.2f)", POSITION.x, POSITION.y, POSITION.z);
	g_pTxtHelper->DrawTextLine(message);
	g_pTxtHelper->End();

}

//--------------------------------------------------------------------------------------
// Release D3D11 resources created in OnD3D11ResizedSwapChain 
//--------------------------------------------------------------------------------------
void CALLBACK OnD3D11ReleasingSwapChain(void* pUserContext)
{
	g_DialogResourceManager.OnD3D11ReleasingSwapChain();
}

//--------------------------------------------------------------------------------------
// Release D3D11 resources created in OnD3D11CreateDevice 
//--------------------------------------------------------------------------------------
void CALLBACK OnD3D11DestroyDevice(void* pUserContext)
{
	g_DialogResourceManager.OnD3D11DestroyDevice();
	DXUTGetGlobalResourceCache().OnDestroyDevice();
	SAFE_DELETE(g_pTxtHelper);


	SAFE_RELEASE(g_pVertexBuffer);
	SAFE_RELEASE(g_pIndexBuffer);
	SAFE_RELEASE(g_pVertexLayout);
	for (int i = 0; i < 6; i++)
		SAFE_RELEASE(g_pSkyboxeRV[i]);
	SAFE_RELEASE(g_pVertexShader);
	SAFE_RELEASE(g_pPixelShader);
	SAFE_RELEASE(g_pCBChangesEveryFrame);
	SAFE_RELEASE(g_pSamplerLinear);
	SAFE_RELEASE(g_pResource);

	dpDestroyContextD3D11(CTX);
}

//--------------------------------------------------------------------------------------
// Handle messages to the application
//--------------------------------------------------------------------------------------
LRESULT CALLBACK MsgProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam, bool* pbNoFurtherProcessing, void* pUserContext)
{
	return 0;
}

//--------------------------------------------------------------------------------------
// Handle key presses
//--------------------------------------------------------------------------------------
void CALLBACK OnKeyboard(UINT nChar, bool bKeyDown, bool bAltDown, void* pUserContext)
{
	if (bKeyDown)
	{
		switch (nChar)
		{
		case VK_F1: // Change as needed                
			break;
		//case 0x30: // 0
		//	g_fHeading -= 10.f;
		//	break;
		//case 0x31: // 1
		//	g_fHeading += 10.f;
		//	break;
		//case 0x32: // 2
		//	g_fPitch -= 10.f;
		//	break;
		//case 0x33: // 3
		//	g_fPitch += 10.f;
		//	break;
		case VK_LEFT:
		case 0x41: // a
			POSITION.x -= 10.0;
			break;
		case VK_RIGHT:
		case 0x44: // d
			POSITION.x += 10.0;
			break;
		case VK_UP:
			POSITION.y += 10.0;
			break;
		case VK_DOWN:
			POSITION.y -= 10.0;
			break;
		case 0x57: // w
			POSITION.z -= 10.0;
			break;
		case 0x53: // s
			POSITION.z += 10.0;
			break;
        case 0x31: // 1
            g_fWarping = std::max(g_fWarping - 0.1f, 0.f);
            break;
        case 0x32: // 2
            g_fWarping = std::min(g_fWarping + 0.1f, 1.f);
            break;
        case 0x33: // 3
            g_fBlending = std::max(g_fBlending - 0.1f, 0.f);
            break;
        case 0x34: // 4
            g_fBlending = std::min(g_fBlending + 0.1f, 1.f);
            break;
        case 0x35: // 5
            g_fBlackLevel = std::max(g_fBlackLevel - 0.1f, 0.f);
            break;
        case 0x36: // 6
            g_fBlackLevel = g_fBlackLevel + 0.1f;
            break;
        case 0x37: // 7
            g_bWarping = !g_bWarping;
            break;
        case 0x38: // 8
            g_bBlending = !g_bBlending;
            break;
        case 0x39: // 9
            g_bBlackLevel = !g_bBlackLevel;
            break;
        case 0x30: // 0
            g_bSecondaryBlending = !g_bSecondaryBlending;
            break;
		}
	}
}

//--------------------------------------------------------------------------------------
// Call if device was removed.  Return true to find a new device, false to quit
//--------------------------------------------------------------------------------------
bool CALLBACK OnDeviceRemoved(void* pUserContext)
{
	return true;
}


//--------------------------------------------------------------------------------------
// Initialize everything and go into a render loop
//--------------------------------------------------------------------------------------
int WINAPI wmain(int argc, wchar_t const* argv[] )
{
	if (NULL != argv)
	{
		for (int i = 1; i < argc; i++)
		{
			std::wstring p(argv[i]);
			if (p[0] == '-')
			{
				std::wstring param = p.substr(1, std::wstring::npos);
				if (param == L"x" && i < argc - 1)
				{
					g_windowPosX = _wtoi(argv[i + 1]);
					i++;
				}
				if (param == L"y" && i < argc - 1)
				{
					g_windowPosY = _wtoi(argv[i + 1]);
					i++;
				}
				if (param == L"w" && i < argc - 1)
				{
					g_windowWidth = _wtoi(argv[i + 1]);
					i++;
				}
				if (param == L"h" && i < argc - 1)
				{
					g_windowHeight = _wtoi(argv[i + 1]);
					i++;
				}
				if (param == L"c" && i < argc - 1)
				{
					g_channel = _wtoi(argv[i + 1]);
					i++;
				}
			}
			else
			{
				g_configfile = std::filesystem::path( argv[i] ).string();
			}
		}
	}

	// Enable run-time memory check for debug builds.
#ifdef _DEBUG
	_CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
#endif

	// DXUT will create and use the best device
	// that is available on the system depending on which D3D callbacks are set below

	// Set general DXUT callbacks
	DXUTSetCallbackFrameMove(OnFrameMove);
	DXUTSetCallbackKeyboard(OnKeyboard);
	DXUTSetCallbackMsgProc(MsgProc);
	DXUTSetCallbackDeviceChanging(ModifyDeviceSettings);
	DXUTSetCallbackDeviceRemoved(OnDeviceRemoved);

	// Set the D3D11 DXUT callbacks. Remove these sets if the app doesn't need to support D3D11
	DXUTSetCallbackD3D11DeviceAcceptable(IsD3D11DeviceAcceptable);
	DXUTSetCallbackD3D11DeviceCreated(OnD3D11CreateDevice);
	DXUTSetCallbackD3D11SwapChainResized(OnD3D11ResizedSwapChain);
	DXUTSetCallbackD3D11FrameRender(OnD3D11FrameRender);
	DXUTSetCallbackD3D11SwapChainReleasing(OnD3D11ReleasingSwapChain);
	DXUTSetCallbackD3D11DeviceDestroyed(OnD3D11DestroyDevice);

	// Perform any application-level initialization here
	DXUTInit(true, true, nullptr); // Parse the command line, show msgboxes on error, no extra command line params
	DXUTSetCursorSettings(true, true); // Show the cursor and clip it when in full screen
	DXUTCreateWindow(L"dpLib testcase");

	// Only require 10-level hardware or later
	DXUTCreateDevice(D3D_FEATURE_LEVEL_10_0, true, g_windowWidth, g_windowHeight);

    DWORD dwStyle = GetWindowLong(DXUTGetHWND(), GWL_STYLE);
    SetWindowLong(DXUTGetHWND(), GWL_STYLE, dwStyle & ~WS_OVERLAPPEDWINDOW);
    BOOL b = SetWindowPos(DXUTGetHWND(), HWND_TOP, g_windowPosX, g_windowPosY, g_windowWidth, g_windowHeight, SWP_NOOWNERZORDER | SWP_FRAMECHANGED);

	DXUTMainLoop(); // Enter into the DXUT render loop

	// Perform any application-level cleanup here
	return DXUTGetExitCode();
}
